// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import React from "react";
import { addUser, removeUser, updateUser, userExists } from "./revpi-nodered.settings.js";
import {
    Button, Dropdown, DropdownItem, DropdownList,
    Flex,
    FlexItem,
    Form,
    FormGroup, FormHelperText,
    List,
    ListItem, MenuToggle,
    Modal,
    Text,
    TextContent, TextInput
} from "@patternfly/react-core";
import { TrashIcon } from "@patternfly/react-icons";
import cockpit from "cockpit";

const _ = cockpit.gettext;

export const UserManagement = ({ users, setUsers, updateSettingsChanges, closeModal, isModalOpen }) => {
    const [modalState, setModalState] = React.useState({
        add: false,
        remove: { open: false, username: null },
        reset: { open: false, username: null }
    });

    const openModal = (type, username = null) => {
        setModalState(prev => ({
            ...prev,
            [type]: typeof prev[type] === "object" ? { open: true, username } : true
        }));
    };

    const closeModalType = (type) => {
        setModalState(prev => ({
            ...prev,
            [type]: typeof prev[type] === "object" ? { open: false, username: null } : false
        }));
    };

    const handlePermissionChange = async (username, permissions) => {
        await updateUser(username, { permissions });
        setUsers(users.map(u => u.username === username ? { ...u, permissions } : u));
        updateSettingsChanges();
    };

    return (
        <>
            <UserList
                users={users}
                onResetPassword={(username) => openModal("reset", username)}
                onRemoveUser={(username) => openModal("remove", username)}
                onEditPermissions={handlePermissionChange}
            />
            <UserModals
                modalState={modalState}
                onClose={closeModalType}
                users={users}
                setUsers={setUsers}
                updateSettingsChanges={updateSettingsChanges}
                isAddUserOpen={isModalOpen}
                closeAddUser={closeModal}
            />
        </>
    );
};

export const UserForm = React.forwardRef(({ onSubmit }, ref) => {
    const inputUsernameRef = React.useRef(null);
    const inputPasswordRef = React.useRef(null);
    const inputPasswordConfirmRef = React.useRef(null);

    const [dropdownValue, setDropdownValue] = React.useState(null);
    const [selectedText, setSelectedText] = React.useState(null);
    const [isOpen, setIsOpen] = React.useState(false);

    const [errors, setErrors] = React.useState({
        username: null,
        password: null,
        confirm: null,
        permissions: null
    });

    const onToggleClick = () => setIsOpen(!isOpen);

    const onSelect = (event, value) => {
        setDropdownValue(value);
        setSelectedText(value === "*" ? _("All") : _("Read"));
        setIsOpen(false);
        setErrors(prev => ({
            ...prev,
            permissions: null
        }));
    };

    const handleSubmit = async () => {
        const username = inputUsernameRef.current.value.trim();
        const password = inputPasswordRef.current.value;
        const confirm = inputPasswordConfirmRef.current.value;

        let hasError = false;
        const newErrors = {
            username: null,
            password: null,
            confirm: null,
            permissions: null
        };

        if (!username) {
            newErrors.username = _("Username is required");
            hasError = true;
        }

        if (!password) {
            newErrors.password = _("Password is required");
            hasError = true;
        }

        if (!confirm) {
            newErrors.confirm = _("Confirm password");
            hasError = true;
        } else if (password !== confirm) {
            newErrors.confirm = _("Passwords do not match");
            hasError = true;
        }

        if (!dropdownValue) {
            newErrors.permissions = _("Select permission level");
            hasError = true;
        }

        if (hasError) {
            setErrors(newErrors);
            return;
        }

        try {
            await onSubmit({
                username,
                password,
                permissions: dropdownValue
            });
        } catch (error) {
            console.error("User submit failed:", error);
            // optionally show global error
        }
    };

    React.useImperativeHandle(ref, () => {
        return {
            submit () {
                handleSubmit();
            },
            clear () {
                if (inputUsernameRef.current) inputUsernameRef.current.value = "";
                if (inputPasswordRef.current) inputPasswordRef.current.value = "";
                if (inputPasswordConfirmRef.current) inputPasswordConfirmRef.current.value = "";
                setDropdownValue(null);
                setSelectedText(null);
                setErrors({});
            }
        };
    });

    return (
        <Form isHorizontal style={{ maxWidth: "400px" }}>
            <FormGroup
                label={_("Username")}
                isRequired
                fieldId='username'
                validated={errors.username ? "error" : "default"}
            >
                <TextInput
                    ref={inputUsernameRef}
                    type='text'
                    id='username'
                    name='username'
                    placeholder={_("Username")}
                    validated={errors.username ? "error" : "default"}
                />
                {errors.username && (
                    <FormHelperText>{errors.username}</FormHelperText>
                )}
            </FormGroup>

            <FormGroup
                label={_("Password")}
                isRequired
                fieldId='password'
                validated={errors.password ? "error" : "default"}
            >
                <TextInput
                    ref={inputPasswordRef}
                    type='password'
                    id='password'
                    name='password'
                    placeholder={_("Password")}
                    validated={errors.password ? "error" : "default"}
                />
                {errors.password && (
                    <FormHelperText>{errors.password}</FormHelperText>
                )}
            </FormGroup>

            <FormGroup
                label={_("Confirm password")}
                isRequired
                fieldId='confirmPassword'
                validated={errors.confirm ? "error" : "default"}
            >
                <TextInput
                    ref={inputPasswordConfirmRef}
                    type='password'
                    id='confirmPassword'
                    name='confirmPassword'
                    placeholder={_("Confirm password")}
                    validated={errors.confirm ? "error" : "default"}
                />
                {errors.confirm && (
                    <FormHelperText>{errors.confirm}</FormHelperText>
                )}
            </FormGroup>

            <FormGroup
                label={_("Permissions")}
                isRequired
                fieldId='permissions'
                validated={errors.permissions ? "error" : "default"}
            >
                <Dropdown
                    isOpen={isOpen}
                    onSelect={onSelect}
                    onOpenChange={open => setIsOpen(open)}
                    toggle={toggleRef => (
                        <MenuToggle
                            ref={toggleRef}
                            onClick={onToggleClick}
                            isExpanded={isOpen}
                        >
                            {selectedText || _("Permission")}
                        </MenuToggle>
                    )}
                    shouldFocusToggleOnSelect
                >
                    <DropdownList>
                        <DropdownItem key='read' value='read'>
                            {_("Read")}
                        </DropdownItem>
                        <DropdownItem key='all' value='*'>
                            {_("All")}
                        </DropdownItem>
                    </DropdownList>
                </Dropdown>
                {errors.permissions && (
                    <FormHelperText>{errors.permissions}</FormHelperText>
                )}
            </FormGroup>

        </Form>
    );
});

const UserList = ({ users, onResetPassword, onRemoveUser, onEditPermissions }) => (
    <List isPlain style={{ marginTop: "1rem" }}>
        {users
            .slice() // prevent mutating the original array
            .sort((a, b) => a.username.localeCompare(b.username))
            .map((user) => (
                <ListItem
                    key={user.username}
                    style={{
                        padding: "1rem",
                        borderTop: "1px solid var(--pf-v5-global--BorderColor--100)"
                    }}
                >
                    <Flex
                        justifyContent={{ default: "justifyContentSpaceBetween" }}
                        alignItems={{ default: "alignItemsCenter" }}
                    >
                        <FlexItem>
                            <strong>{user.username}</strong>
                        </FlexItem>
                        <FlexItem>
                            <PermissionDropdown
                                value={user.permissions}
                                onChange={(newPerm) => onEditPermissions(user.username, newPerm)}
                            />
                            <Button
                                variant='secondary'
                                onClick={() => onResetPassword(user.username)}
                                style={{ marginRight: "1rem" }}
                            >
                                {_("Reset Password")}
                            </Button>
                            <Button
                                variant='danger'
                                icon={<TrashIcon />}
                                onClick={() => onRemoveUser(user.username)}
                            >
                                {_("Remove")}
                            </Button>
                        </FlexItem>
                    </Flex>
                </ListItem>
            ))}
    </List>
);

const UserModals = ({
    modalState,
    onClose,
    users,
    setUsers,
    updateSettingsChanges,
    isAddUserOpen,
    closeAddUser
}) => {
    return (
        <>
            <AddUserModal
                isOpen={isAddUserOpen}
                onClose={closeAddUser}
                users={users}
                setUsers={setUsers}
                updateSettingsChanges={updateSettingsChanges}
            />
            <RemoveUserModal
                isOpen={modalState.remove.open}
                username={modalState.remove.username}
                onClose={() => onClose("remove")}
                users={users}
                setUsers={setUsers}
                updateSettingsChanges={updateSettingsChanges}
            />
            <ResetPasswordModal
                isOpen={modalState.reset.open}
                username={modalState.reset.username}
                onClose={() => onClose("reset")}
                updateSettingsChanges={updateSettingsChanges}
            />
        </>
    );
};

const AddUserModal = ({ isOpen, onClose, users, setUsers, updateSettingsChanges }) => {
    const formRef = React.useRef();

    const handleAdd = async (data) => {
        if (userExists(data.username)) return;
        await addUser(data.username, data.password, data.permissions);
        setUsers([...users, { username: data.username, permissions: data.permissions }]);
        updateSettingsChanges();
        onClose();
    };

    return (
        <Modal
            title={_("Add New User")}
            isOpen={isOpen}
            onClose={onClose}
            variant='small'
            actions={[
                <Button key='close' variant='secondary' onClick={onClose}>{_("Close")}</Button>,
                <Button key='submit' variant='primary' onClick={() => formRef.current.submit()}>{_("Add User")}</Button>
            ]}
        >
            <UserForm ref={formRef} onSubmit={handleAdd} />
        </Modal>
    );
};

const RemoveUserModal = ({ isOpen, username, onClose, users, setUsers, updateSettingsChanges }) => {
    const handleRemove = async () => {
        await removeUser(username);
        setUsers(users.filter(u => u.username !== username));
        updateSettingsChanges();
        onClose();
    };

    return (
        <Modal
            title={_("Confirm Removal")}
            isOpen={isOpen}
            onClose={onClose}
            variant='small'
            actions={[
                <Button key='cancel' variant='secondary' onClick={onClose}>{_("Cancel")}</Button>,
                <Button key='confirm' variant='danger' onClick={handleRemove}>{_("Remove")}</Button>
            ]}
        >
            <TextContent>
                <Text>{_("Are you sure you want to remove user")} <strong>{username}</strong>?</Text>
            </TextContent>
        </Modal>
    );
};

const ResetPasswordModal = ({ isOpen, username, onClose, updateSettingsChanges }) => {
    const [error, setError] = React.useState(null);
    const passwordRef = React.useRef(null);
    const confirmRef = React.useRef(null);

    const handleReset = async () => {
        const password = passwordRef.current.value;
        const confirm = confirmRef.current.value;
        const validationError = validatePasswords(password, confirm);

        if (validationError) return setError(validationError);

        try {
            await updateUser(username, { password });
            updateSettingsChanges();
            onClose();
            setError(null);
        } catch (err) {
            setError("Failed to reset password.");
        }
    };

    return (
        <Modal
            title={_("Reset Password")}
            isOpen={isOpen}
            onClose={onClose}
            variant='small'
            actions={[
                <Button key='cancel' variant='secondary' onClick={onClose}>{_("Cancel")}</Button>,
                <Button key='reset' variant='primary' onClick={handleReset}>{_("Save")}</Button>
            ]}
        >
            <TextContent>
                <Text>{_("Reset password for ")} <strong>{username}</strong></Text>
            </TextContent>

            <Form style={{ marginTop: "1rem" }}>
                <FormGroup label={_("New Password")} isRequired fieldId='reset-password'>
                    <TextInput ref={passwordRef} isRequired type='password' id='reset-password' />
                </FormGroup>

                <FormGroup label={_("Confirm password")} isRequired fieldId='reset-confirm'>
                    <TextInput ref={confirmRef} isRequired type='password' id='reset-confirm' />
                </FormGroup>

                {error && <FormHelperText isError>{error}</FormHelperText>}
            </Form>
        </Modal>
    );
};

/**
 * A small inline dropdown to switch between `read` and `*` permissions.
 * @param {object} props
 * @param {"read"|"*"} props.value - Current selected permission
 * @param {Function} props.onChange - Callback with new value
 * @returns {JSX.Element}
 */
export const PermissionDropdown = ({
    value,
    onChange
}) => {
    const [isOpen, setIsOpen] = React.useState(false);

    const displayText = value === "*" ? _("All") : _("Read");

    const handleSelect = (event, newValue) => {
        setIsOpen(false);
        onChange(newValue);
    };

    return (
        <Dropdown
            isOpen={isOpen}
            onSelect={handleSelect}
            onOpenChange={setIsOpen}
            toggle={(toggleRef) => (
                <MenuToggle ref={toggleRef} onClick={() => setIsOpen(!isOpen)} isExpanded={isOpen} style={{ marginRight: "1rem" }}>
                    {displayText}
                </MenuToggle>
            )}
            shouldFocusToggleOnSelect
        >
            <DropdownItem value='read' key='read'>{_("Read")}</DropdownItem>
            <DropdownItem value='*' key='all'>{_("All")}</DropdownItem>
        </Dropdown>
    );
};

function validatePasswords (password, confirm) {
    if (!password || !confirm) return "Password and confirmation are required.";
    if (password !== confirm) return "Passwords do not match.";
    return null;
}
