// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";
import { fileExists, isUsingAvahiConnection, readFromFile, writeToFile } from "../../common/helper.js";

// consts
const SUDOERS_FILE_PATH = "/etc/sudoers.d/050_sudo-group-password-prompt";

/**
 * Executes a revpi-config command using Cockpit's `spawn` function.
 *
 * @param {string} command - The revpi-config command to execute (e.g., "enable", "disable", "availstat").
 * @param {string[]} [ids=[]] - An optional array of feature IDs to pass as arguments.
 * @returns {Promise<string>} - A promise that resolves with the command output.
 *
 * This function:
 * - Calls `/usr/bin/revpi-config` with the provided command and IDs.
 * - Runs the command with superuser privileges (`superuser: "require"`).
 * - Can be used to enable, disable, or check the status of revpi-config features.
 * - The resolved output can be parsed for further processing.
 */
const callRevpiConfig = async (command, ids = []) => {
    return cockpit.spawn(["/usr/bin/revpi-config", command, ...ids], { superuser: "require" });
};

/**
 * Translates revpi-config response status codes into human-readable states.
 *
 * @param {string} id - The feature ID for debugging purposes.
 * @param {string[]} response - The response array containing status codes.
 * @returns {string} - A human-readable status string.
 */
const getStatus = (id, response) => {
    // If response contains only one status code, map it to a readable format
    if (response.length === 1) {
        switch (response[0]) {
        case "0":
            return "disabled"; // Feature is turned off
        case "1":
            return "enabled"; // Feature is active
        case "2":
            return "unavailable"; // Feature is not supported or inaccessible
        default:
            console.warn(`Unexpected response for ${id}: ${response}. Expected 0, 1, or 2.`);
            return "unknown"; // Return a default fallback for unexpected values
        }
    }

    // If multiple values exist, return the raw response for further processing
    return response[0];
};

const disableFeature = (item) => {
    if (item.id === "avahi") {
        if (isUsingAvahiConnection()) {
            return true;
        }
    }
    // all other features are enabled
    return false;
};

/**
 * Extends the data from RevPiConfigData with UI-specific variables
 * that are used to properly display feature states in app.jsx.
 */
const mapFeatureData = (item, status) => ({
    ...item, // Keep original item properties
    status, // Parsed status from response
    isEnabled: status === "enabled", // Determines if the feature is active
    isAvailable: status !== "unavailable", // Checks if the feature is available

    // UI-only flag: Determines if the switch should be visually disabled
    // (This is NOT related to the revpi-config status)
    isDisabled: disableFeature(item),

    hasError: false, // Default: No error state
    isLoading: false // Default: Not loading
});

/**
 * Parses the revpi-config response data and converts it into
 * a JavaScript object that can be used throughout the app
 * to display revpi-config features.
 */
const parseResponseToObject = (responseString, configData) => {
    /**
     * Extracts a status value (or multiple values) for a given feature
     * from the response array, considering index shifts.
     */
    const extractStatus = (responseArray, itemIndex, indexShifts, responseCount) => {
        return responseArray.slice(itemIndex + indexShifts, itemIndex + indexShifts + responseCount);
    };

    const responseArray = responseString.trim().split(" "); // Convert response string into an array
    let indexShifts = 0; // Tracks shifts in the index when parsing multiple response values

    /**
     * Iterates through configData and extracts the corresponding status
     * values from the response array.
     */
    const statuses = configData.map((item, i) => {
        const responseItems = item.responseCount || 1; // Number of expected response values
        const rawStatus = extractStatus(responseArray, i, indexShifts, responseItems); // Get the raw status
        const status = getStatus(item.id, rawStatus); // Convert raw status to readable format

        indexShifts += responseItems - 1; // Adjust index shifts when processing multiple values
        return { item, status }; // Return the mapped item with its parsed status
    });

    /**
     * Maps the extracted status values into feature data objects
     * that contain UI-related properties.
     */
    const parsedResponse = statuses.map(({ item, status }) => mapFeatureData(item, status));

    /**
     * Validates that the total extracted values match the expected configData length.
     * If not, there may be an issue with the response parsing logic.
     */
    if (responseArray.length - indexShifts !== configData.length) {
        throw new Error("Error: An issue occurred with the index shifting logic.");
    }
    return parsedResponse;
};

export const toggleConfiguration = async (id, isEnabled) => {
    const command = isEnabled ? "disable" : "enable";
    return callRevpiConfig(command, [id]);
};

export const getRevPiConfigStatus = async (configData) => {
    const ids = configData.map((item) => item.id);
    const response = await callRevpiConfig("availstat", ids);
    return parseResponseToObject(response, configData);
};

export const hasLowMemory = async () => {
    const minimumMemory = 1536;
    try {
        const output = await cockpit.spawn(["free", "-m"]);
        const match = output.match(/^Mem:\s+(\d+)/m);

        if (match) {
            const ramMb = parseInt(match[1], 10);
            return ramMb <= minimumMemory; // Return true if 1536 MB or less, false otherwise
        }

        console.error("Could not determine RAM size.");
        return false; // Default to false if parsing fails
    } catch (error) {
        console.error("Error checking RAM size:", error);
        return false; // Return false in case of error
    }
};

/**
 * Checks whether a password is required for sudo commands.
 *
 * This asynchronous function determines if the system requires a sudo password
 * based on the presence and content of a sudoers file. If the file exists, it
 * checks for the "NOPASSWD:" directive within the content to ascertain if a
 * password is bypassed. If the file does not exist, it updates the system configuration
 * to unset password requirements and returns false.
 *
 * @returns {Promise<boolean>} A promise that resolves to `true` if a password is
 * required for sudo commands, or `false` otherwise.
 */
export const isSudoPasswordRequired = async () => {
    const isFileExists = await fileExists(SUDOERS_FILE_PATH);
    if (isFileExists) {
        const fileContent = await readFromFile(SUDOERS_FILE_PATH);
        return !fileContent.includes("NOPASSWD:");
    } else {
        await unsetPasswordRequiredForSudo();
        return false;
    }
};

/**
 * Asynchronously updates the sudoers file to require a password for using sudo.
 *
 * This function modifies the sudoers configuration, enforcing that a password
 * is required for any sudo actions. It writes a predefined configuration string
 * to the sudoers file.
 *
 * @returns {Promise<void>} A promise that resolves when the file operation is complete.
 *                          Rejections may occur if there are issues writing to the file.
 */
export const setPasswordRequiredForSudo = async () => {
    const fileContent = "%sudo ALL=(ALL) ALL";
    return writeToFile(SUDOERS_FILE_PATH, fileContent);
};

/**
 * Disables the password requirement for sudo commands.
 *
 * This asynchronous function modifies the sudoers file by adding a rule that
 * allows all users under the sudo group to execute sudo commands without
 * being prompted for a password. The function writes a predefined configuration
 * string to the appropriate file to achieve this behavior.
 *
 * @function unsetPasswordRequiredForSudo
 * @async
 * @returns {Promise<void>} A promise that resolves when the sudoers file
 * is successfully updated, or rejects if an error occurs during file writing.
 */
export const unsetPasswordRequiredForSudo = async () => {
    const fileContent = "%sudo ALL=(ALL) NOPASSWD: ALL";
    return writeToFile(SUDOERS_FILE_PATH, fileContent);
};
