// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import { Alert, Card, CardBody, CardTitle, Grid, GridItem, Switch } from "@patternfly/react-core";
import React from "react";
import cockpit from "cockpit";
import {
    isSudoPasswordRequired,
    setPasswordRequiredForSudo,
    unsetPasswordRequiredForSudo
} from "../revpi-config/revpi-config-service.jsx";
import { RevPiFlexBase } from "../revpi-config.components.jsx";
import { SecurityIcon, WrenchIcon } from "@patternfly/react-icons";
import { UiDivider, UiCardTitle } from "../../common/ui-components.jsx";
import { useBasicModal } from "../../common/hooks.jsx";

const _ = cockpit.gettext;

export function SystemConfigurationCard ({ getRevPiSwitch, getRevPiToggle }) {
    const cardTitle = _("System Configuration");
    const [sudoPasswordRequired, setSudoPasswordRequired] = React.useState(true);

    const {
        BasicModal,
        handleModalToggle
    } = useBasicModal(
        {
            title: <><SecurityIcon style={{ marginRight: "1rem" }} /> {_("Deactivate sudo password prompt?")}</>,
            message: _("Are you sure you want to disable password prompt for sudo commands? This can pose a security risk as commands requiring superuser privileges can be executed without additional authentication."),
            confirmCallback: async () => {
                await unsetPasswordRequiredForSudo();
                setSudoPasswordRequired(false);
            },
            cancelText: _("Cancel"),
            confirmText: _("Confirm"),
            confirmVariant: "danger",
            modalSize: "medium"
        }
    );
    React.useEffect(() => {
        async function fetchSudoPasswordRequired () {
            const passwordRequired = await isSudoPasswordRequired();
            setSudoPasswordRequired(passwordRequired);
        }
        fetchSudoPasswordRequired();
    }, []);

    const handlePasswordRequiredToggle = async () => {
        if (sudoPasswordRequired) {
            handleModalToggle();
            return;
        }
        // Re-activate the password requirement for sudo commands
        // Deactivation is handled in modal confirmCallback
        await setPasswordRequiredForSudo();
        setSudoPasswordRequired(true);
    };

    return (
        <Card isFullHeight name='system-configuration'>
            <CardTitle>
                <UiCardTitle>
                    <WrenchIcon style={{ marginRight: "1rem" }} />
                    {cardTitle}
                </UiCardTitle>
                <UiDivider />
            </CardTitle>
            <CardBody>
                <Grid hasGutter>
                    <GridItem span={12}>{getRevPiSwitch("gui")}</GridItem>
                    <GridItem span={12}>{getRevPiSwitch("dphys-swapfile")}</GridItem>
                    <GridItem span={12}>{getRevPiSwitch("avahi")}</GridItem>
                    <GridItem span={12}>{getRevPiSwitch("systemd-timesyncd")}</GridItem>
                    <GridItem span={12}>{getRevPiToggle("var-log.mount")}</GridItem>
                    <GridItem span={12}>{getRevPiSwitch("ssh")}</GridItem>
                    <GridItem span={12}>
                        <RevPiFlexBase item={{
                            title: _("Require password for sudo commands"),
                            helpText: _("When disabled, sudo commands can be executed without additional password prompts."),
                            hasError: false,
                            isAvailable: true,
                            isLoading: false,
                            rebootRequired: false
                        }}
                        >
                            <Switch
                                hasCheckIcon
                                label={_("Enabled")}
                                labelOff={_("Disabled")}
                                isChecked={sudoPasswordRequired}
                                onChange={handlePasswordRequiredToggle}
                            />
                        </RevPiFlexBase>
                    </GridItem>
                    <GridItem
                        span={12}
                        style={{
                            display: sudoPasswordRequired ? "none" : "block"
                        }}
                    >
                        <Alert
                            isInline
                            variant='danger'
                            title={_("Security Risk")}
                        >{_("With sudo password disabled, commands that require superuser privileges can be executed without additional password prompts.")}
                        </Alert>
                    </GridItem>
                </Grid>
            </CardBody>
            {BasicModal}
        </Card>
    );
}
