// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import { WithDialogs } from "../../pkg/lib/dialogs.jsx";
import { EmptyStatePanel } from "../../pkg/lib/cockpit-components-empty-state.jsx";
import { SuperuserButton } from "../../pkg/shell/superuser.jsx";
import React, { createContext, useCallback, useContext, useEffect, useState } from "react";
import { Button, Flex, FlexItem, Modal, ModalVariant } from "@patternfly/react-core";
import { ModalBody, ModalFooter, ModalHeader } from "@patternfly/react-core/next";
// eslint-disable-next-line camelcase
import { isFirewallInstalled, isPackageInstalled, reload_bridge_packages } from "./helper.js";
import { installEventEmitter } from "../revpi-config/emitters.js";
// eslint-disable-next-line camelcase
import { install_dialog } from "../../pkg/lib/cockpit-components-install-dialog.jsx";

import cockpit from "cockpit";

/**
 * A reference to the `gettext` function from the Cockpit framework,
 * used for internationalization and localization in applications.
 * This function is typically utilized to translate strings into the
 * user's preferred language.
 *
 * The `gettext` function takes a string as an argument and returns
 * its translated version based on the current locale settings.
 *
 * Commonly employed for marking strings in applications that are
 * intended to be translated, allowing dynamic language support.
 */
const _ = cockpit.gettext;

/**
 * A functional component that renders an interface requiring administrative access.
 * It displays a message indicating that administrative privileges are required for the use of a specific plugin.
 * Includes an actionable button for superuser access.
 *
 * @return {JSX.Element} The rendered component, containing a dialog with a title, description, and a superuser access button.
 */
export function AdminAccessButton () {
    return (
        <WithDialogs>
            <EmptyStatePanel
                title={_("Administrative access required")}
                paragraph={_("Administrative access is required for using this plugin.")}
                action={<SuperuserButton />}
            />
        </WithDialogs>
    );
}

/**
 * A custom hook that provides a basic modal component and its associated functionalities.
 *
 * @param {Object} options - Configuration options for the modal.
 * @param {string} options.title - The title of the modal.
 * @param {string} options.message - The message or content to be displayed within the modal body.
 * @param {Function} options.confirmCallback - The callback function executed when the confirm button is clicked.
 * @param {string} options.confirmText - The text displayed on the confirm button.
 * @param {string} options.cancelText - The text displayed on the cancel button.
 * @param {string} [options.confirmVariant="primary"] - The visual variant of the confirm button (e.g., "primary").
 * @param {string} [options.cancelVariant="secondary"] - The visual variant of the cancel button (e.g., "secondary").
 * @param {string} [options.modalSize="small"] - The size of the modal (e.g., "small", "medium", "large").
 *
 * @returns {Object} - Returns an object with the following properties:
 * - `BasicModal`: A React functional component representing the modal.
 * - `handleModalToggle`: A function to toggle the visibility of the modal.
 * - `isModalOpen`: A boolean indicating whether the modal is open or closed.
 */
export const useBasicModal = ({
    title,
    message,
    confirmCallback,
    confirmText,
    cancelText,
    confirmVariant = "primary",
    cancelVariant = "secondary",
    modalSize = "small"
}) => {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const handleModalToggle = _event => {
        setIsModalOpen(!isModalOpen);
    };

    const BasicModal =
        (
            <>
                <Modal
                    isOpen={isModalOpen}
                    onClose={handleModalToggle}
                    variant={ModalVariant[modalSize]}
                >
                    <ModalHeader title={title} />
                    <ModalBody>
                        {message}
                    </ModalBody>
                    <ModalFooter>
                        <Flex
                            style={{ width: "100%" }} direction={{ default: "row" }}
                            justifyContent={{ default: "justifyContentFlexEnd" }}
                        >
                            <FlexItem>
                                <Button
                                    key='cancel'
                                    onClick={handleModalToggle}
                                    variant={cancelVariant}
                                >{cancelText}
                                </Button>
                            </FlexItem>
                            <FlexItem>
                                <Button
                                    key='confirm'
                                    onClick={() => {
                                        confirmCallback();
                                        handleModalToggle();
                                    }}
                                    variant={confirmVariant}
                                >{confirmText}
                                </Button>
                            </FlexItem>
                        </Flex>
                    </ModalFooter>
                </Modal>
            </>

        );
    return {
        BasicModal,
        handleModalToggle,
        isModalOpen
    };
};

/**
 * A custom hook that manages and returns the admin permission state.
 *
 * This hook checks if the user has administrative privileges by
 * interacting with the `cockpit.permission` API. It listens for
 * changes to the permission state and updates the admin status accordingly.
 *
 * @returns {boolean} `true` if the user has admin permissions, otherwise `false`.
 */
export const useAdminPermission = () => {
    const [isAdmin, setIsAdmin] = useState(false);

    useEffect(() => {
        const permission = cockpit.permission({ admin: true });

        const handlePermissionChange = () => {
            setIsAdmin(permission.allowed);
        };

        permission.addEventListener("changed", handlePermissionChange);

        // Initial check
        handlePermissionChange();

        return () => {
            permission.removeEventListener("changed", handlePermissionChange);
        };
    }, []);

    return isAdmin;
};
/**
 * Custom hook to access the NotificationContext.
 * This hook provides an interface to interact with the notification system,
 * allowing components to consume notification-related state or dispatch actions.
 *
 * @function
 * @returns {Object} The value provided by the NotificationContext, which
 *          typically includes notification state and methods to manage
 *          notifications.
 */

/**
 * Represents a function used to manage and configure app-related settings and behavior.
 * This function returns an object that encapsulates the application's configuration details,
 * installation logic, and related metadata.
 *
 * @param {Object} params An object containing configuration parameters for the application.
 * @param {string} params.name The internal name or identifier of the application.
 * @param {string} params.displayName The user-friendly display name for the application.
 * @param {Array<string>} params.packageNames A list of package names associated with the application.
 * @param {string} params.displayType A string representing the display category or type of the application.
 * @param {Function} params.click A callback function to handle user actions or interactions with the application.
 * @param {string} params.helpText Optional help or descriptive text to guide the user.
 *
 * @returns {Object} An object containing the application configuration, including the name, displayName,
 * displayType, click handler, installer instance, and help text.
 */
export const useAppConfig = ({
    name,
    displayName,
    packageNames,
    displayType,
    click,
    helpText
}) => {
    const installer = useFeatureInstaller(displayName, packageNames);
    return {
        name,
        displayName,
        displayType,
        click,
        installer,
        helpText
    };
};

/**
 * Manages the installation of a feature by checking its installation status and providing functionality to perform the installation.
 *
 * @param {string} featureName - The name of the feature to be installed.
 * @param {string[]} packageNames - An array of package names that are required for the feature installation.
 * @return {Object} An object containing the following properties:
 * - `isInstalled` {boolean}: Indicates whether the feature is currently installed.
 * - `isInstalling` {boolean}: Indicates whether the feature is currently being installed.
 * - `handleInstall` {Function}: A function to initiate the installation of the feature.
 */
export function useFeatureInstaller (featureName, packageNames) {
    const [isInstalled, setIsInstalled] = useState(false);
    const [isInstalling, setIsInstalling] = useState(false);
    const { addNotification } = useNotifications();
    const notificationTitle = _("Installation successful");
    const notificationText = _("$0 successfully installed");

    const checkInstallation = useCallback(async () => {
        let installed = true;
        for (const packageName of packageNames) {
            const isPkgInstalled = await isPackageInstalled(packageName);
            installed = installed && isPkgInstalled;
            if (!installed) break;
        }
        setIsInstalled(installed);
        return installed;
    }, [packageNames]);

    const handleInstall = useCallback(async () => {
        installEventEmitter.dispatchEvent(installEventEmitter.events.start);
        setIsInstalling(true);

        try {
            await install_dialog(packageNames);
            await checkInstallation();
            addNotification(notificationTitle, cockpit.format(notificationText, featureName));
        } catch (error) {
            console.error("Installation failed", error);
        } finally {
            setIsInstalling(false);
            reload_bridge_packages();
            installEventEmitter.dispatchEvent(installEventEmitter.events.finish);
        }
    }, [packageNames, checkInstallation, featureName]);

    useEffect(() => {
        checkInstallation();

        const handleStart = () => setIsInstalling(true);
        const handleFinish = () => setIsInstalling(false);

        installEventEmitter.addEventListener(installEventEmitter.events.start, handleStart);
        installEventEmitter.addEventListener(installEventEmitter.events.finish, handleFinish);

        return () => {
            installEventEmitter.removeEventListener(installEventEmitter.events.start, handleStart);
            installEventEmitter.removeEventListener(installEventEmitter.events.finish, handleFinish);
        };
    }, [checkInstallation]);

    return {
        isInstalled,
        isInstalling,
        handleInstall
    };
}

/**
 * A React Context object used for managing notification-related state
 * and behaviors within the application. The `NotificationContext` provides
 * a way to share notification data (such as messages, types, or visibility)
 * and associated functions (such as displaying or dismissing notifications)
 * across the component tree without having to explicitly pass props down
 * at every level.
 *
 * It is designed to be used with a Context Provider component to supply the
 * relevant notification values and actions, and to be consumed by any
 * components that require access to the notification system.
 *
 * Typical use cases for `NotificationContext` include implementing
 * global or localized notifications for user feedback, error messages,
 * or other UI alerts.
 */
const NotificationContext = createContext();
/**
 * `useNotifications` is a custom hook that provides access to the current notification context.
 *
 * This hook abstracts the use of `useContext(NotificationContext)` to offer a cleaner and more
 * modular way to access and manage notification-related data or actions within the application.
 *
 * It simplifies the process of retrieving and interacting with the `NotificationContext`,
 * which may contain functionality for showing, hiding, or managing application-wide notifications.
 *
 * @function
 * @returns {Object} - The current value from the `NotificationContext`, which typically includes
 * notification-related state and actions.
 */
export const useNotifications = () => useContext(NotificationContext);

/**
 * Provides a context for managing notifications, including adding and removing notifications.
 *
 * @param {object} props - The properties object.
 * @param {React.ReactNode} props.children - The child components to render within the provider.
 * @return {JSX.Element} The NotificationContext Provider wrapping the child components.
 */
export function NotificationProvider ({ children }) {
    const [notifications, setNotifications] = useState([]);

    // Function to add a notification
    const addNotification = (title, message, variant = "info") => {
        const newNotification = {
            id: new Date().getTime(),
            variant,
            title,
            message
        };
        setNotifications(prevNotifications => [...prevNotifications, newNotification]);
    };

    // Function to remove a notification
    const removeNotification = (id) => {
        setNotifications(notifications.filter(notification => notification.id !== id));
    };

    return (
        <NotificationContext.Provider value={{
            addNotification,
            removeNotification,
            notifications
        }}
        >
            {children}
        </NotificationContext.Provider>
    );
}

/**
 * Custom Hook to check if the firewalld package is installed
 * and to determine the package names dynamically.
 */
export function useNoderedPackageNames () {
    const [isFirewallInstalledState, setIsFirewallInstalledState] = useState(null);

    useEffect(() => {
        async function checkFirewall () {
            const installed = await isFirewallInstalled();
            setIsFirewallInstalledState(installed);
        }

        checkFirewall();
    }, []);

    const revpiNodeRedPackageNames = [
        "revpi-nodered",
        "revpi-nodered-proxy-apache",
        "noderedrevpinodes-server",
        "cockpit-revpi-nodered",
        ...(isFirewallInstalledState ? ["revpi-nodered-proxy-firewalld"] : [])
    ];

    return revpiNodeRedPackageNames;
}
