// SPDX-FileCopyrightText: 2023-2025 KUNBUS GmbH
//
// SPDX-License-Identifier: GPL-2.0-or-later

import cockpit from "cockpit";

/**
 * The name of the plugin derived from the script's file path.
 * Extracted from the second-to-last segment of the path in the `src` attribute
 * of the currently executing script.
 * Useful for identifying or configuring the plugin based on its directory name.
 *
 * Note: Relies on the current script being properly referenced in `document.currentScript`.
 */
const PLUGIN_NAME = document.currentScript.src.split("/").at(-2);

/**
 * Determines if the current host of the Cockpit transport is "localhost".
 *
 * The function checks the `host` property of the Cockpit transport object
 * and verifies if it matches the string "localhost". If it does, the function
 * returns true, indicating that the current execution is occurring on a local
 * host. Otherwise, it returns false.
 *
 * @returns {boolean} True if the Cockpit transport host is "localhost"; false otherwise.
 */
const isLocalHost = () => cockpit.transport.host === "localhost";

/**
 * Retrieves the host name based on the transport's host configuration.
 *
 * This function checks the `cockpit.transport.host` property and determines
 * the appropriate host name to return. If the `cockpit.transport.host`
 * equals "localhost", it returns the `window.location.hostname`. Otherwise,
 * it extracts and returns the portion of the host string after the "@" symbol.
 *
 * @returns {string} The resolved host name.
 */
export const getHostName = () => {
    return isLocalHost()
        ? window.location.hostname
        : cockpit.transport.host.split("@").pop();
};

/**
 * Retrieves the port number for the application's connection.
 *
 * Depending on whether the application is running on a local host or a remote server,
 * this function determines and returns the appropriate port number.
 *
 * - If the application is running on a local host, it returns the current window location's port.
 *   If no port is explicitly defined, it defaults to 443.
 * - If the application is running on a remote server, it returns the default port 41443.
 *
 * @returns {number} The port number to be used for the application.
 */
export const getPortNumber = () => {
    return isLocalHost()
        ? window.location.port || 443
        : 41443;
};

/**
 * Retrieves the SSH address for a given user.
 *
 * This function generates an SSH address string based on the provided user's
 * name and the hostname returned by the `getHostName` function.
 *
 * @param {Object} user - The cockpit user object containing user details, like name.
 * @returns {string} A formatted SSH address in the form of "username@hostname".
 */
export const getSshAddress = (user) => `${user.name}@${getHostName()}`;

/**
 * Determines whether the current connection is using Avahi (a service discovery system)
 * by checking if the hostname includes the ".local" domain suffix.
 *
 * @returns {boolean} True if the hostname includes ".local", indicating an Avahi connection; otherwise, false.
 */
export const isUsingAvahiConnection = () => getHostName().includes(".local");

/**
 * Asynchronously retrieves the base directory of a plugin by checking known paths for the plugin's manifest file.
 *
 * The method searches through a predefined list of directories for the plugin's manifest.json file and returns the path
 * to the first directory where the manifest is found.
 *
 * @return {Promise<string|undefined>} A promise that resolves to the plugin's base directory if found, or undefined if
 *                                     the manifest file cannot be located in any of the paths.
 */
export async function getPluginBaseDir () {
    const home = await cockpit.script("echo -n ~");
    const paths = [
        `${home}/.local/share/cockpit/${PLUGIN_NAME}`,
        `/usr/local/share/cockpit/${PLUGIN_NAME}`,
        `/usr/share/cockpit/${PLUGIN_NAME}`
    ];
    for (const path of paths) {
        const file = `${path}/manifest.json`;
        const test = cockpit.file(file);
        const res = await test.read();
        if (res !== null) return path;
    }
}

/**
 * Parses relative file paths into absolute plugin paths.
 * Returns absolute file paths without any change
 * @param {string} filePath
 * @returns {string} absolute path
 */
export async function parseFilePath (filePath) {
    if (filePath[0] !== "/") {
        const baseDir = await getPluginBaseDir();
        filePath = `${baseDir}/${filePath}`;
    }
    return filePath;
}

/**
 * Checks if the specified file exists.
 *
 * The method first attempts to check the file as a regular user.
 * If necessary, it then tries to access the file with superuser rights
 * (`superuser: "try"`). This allows access to files that may not be visible
 * without elevated permissions.
 *
 * A return value of `false` can mean two things:
 * 1. The file does not exist.
 * 2. The permissions are insufficient to determine if the file exists.
 *
 * This distinction is important, as a lack of access due to permissions does not
 * conclusively mean the file does not exist.
 *
 * @param {string} filePath - The path of the file to check.
 * @return {Promise<boolean>} A promise that resolves to `true` if the file exists,
 *                            otherwise `false`.
 */
export async function fileExists (filePath) {
    filePath = await parseFilePath(filePath);
    try {
        await cockpit.script(`[ -f "${filePath}" ]`, { superuser: "try" });
        return true;
    } catch (error) {
        return false;
    }
}

/**
 * Checks if the specified directory exists.
 *
 * The method first attempts to check the directory as a regular user.
 * If necessary, it then tries to access the directory with superuser rights
 * (`superuser: "try"`). This allows access to directories that may not be visible
 * without elevated permissions.
 *
 * A return value of `false` can mean two things:
 * 1. The directory does not exist.
 * 2. The permissions are insufficient to determine if the directory exists.
 *
 * This distinction is important, as a lack of access due to permissions does not
 * conclusively mean the directory does not exist.
 *
 * @param {string} dirPath - The path of the directory to check.
 * @return {Promise<boolean>} A promise that resolves to `true` if the directory exists,
 *                            otherwise `false`.
 */
export async function dirExists (dirPath) {
    dirPath = await parseFilePath(dirPath);
    try {
        await cockpit.script(`[ -d "${dirPath}" ]`, { superuser: "try" });
        return true;
    } catch (error) {
        return false;
    }
}

/**
 * Retrieves all TCP ports that a given process is listening on.
 *
 * This function uses `lsof` via `cockpit.script` to list TCP ports in LISTEN state
 * for a given process ID. It extracts and parses the port numbers using `awk`.
 *
 * @param {number|string} pid - The process ID of the target service.
 * @returns {Promise<number[]|undefined>} A promise that resolves to an array of TCP port numbers,
 * or `undefined` if the process has no listening ports or does not exist.
 *
 * @example
 * const ports = await getBindPorts(1234);
 * if (ports) {
 *   console.log("Listening ports:", ports); // e.g., [1880]
 * } else {
 *   console.log("No ports found or process not running.");
 * }
 */
export async function getBindPorts (pid) {
    const port = await cockpit.script(
        `
        lsof -nP -iTCP -sTCP:LISTEN -a -p ${pid} | awk 'NR>1 {
            split($9, a, ":");
            port = a[length(a)];
            print port;
        }'
        `,
        { superuser: true }
    );
    if (port !== "") {
        return port.trim().split("\n").map(port => parseInt(port, 10));
    } else {
        console.log(`No process for ${pid}`);
    }
}

// region D-Bus

/**
 * Triggers a reload of all bridge packages (i.e., Cockpit plugins).
 *
 * This function calls the internal D-Bus method `cockpit.Packages.Reload` on `/packages`,
 * using the internal Cockpit bus. It is equivalent to calling the `reload` function in
 * Cockpit's built-in `packagekit.js`, and is typically used to refresh plugin availability
 * after installing new Cockpit-related packages.
 *
 * @returns {Promise} A promise that resolves once the reload request has been sent.
 *
 * @see https://github.com/cockpit-project/cockpit/blob/main/pkg/lib/packagekit.js
 */
// eslint-disable-next-line camelcase
export function reload_bridge_packages () {
    return cockpit.dbus(null, { bus: "internal" }).call("/packages", "cockpit.Packages", "Reload", []);
}

// endregion

/**
 * Reads content from a file.
 * @param {string} filePath
 * @returns {string|null} File content if exists, otherwise null.
 */
export async function readFromFile (filePath) {
    filePath = await parseFilePath(filePath);
    const file = cockpit.file(filePath,
        { superuser: "try" });
    const content = await file.read();
    file.close();
    return content;
}

/**
 * Writes content to a file at the specified file path, optionally supporting safe write operations.
 *
 * @param {string} filePath - The path to the file where the content will be written.
 * @param {string} newFileContent - The content to write to the file.
 * @param {boolean} [writeSafely=true] - Whether to write the content safely by using a temporary file.
 * @return {Promise<void>} A promise that resolves when the write operation is complete.
 */
export async function writeToFile (filePath, newFileContent, writeSafely = true) {
    filePath = await parseFilePath(filePath);
    const filePathTmp = `${filePath}.tmp`;

    try {
        await cockpit.file(
            writeSafely ? filePathTmp : filePath,
            { superuser: "try" }
        ).replace(newFileContent);
        if (writeSafely) {
            cockpit.script(`mv "${filePathTmp}" "${filePath}"`, { superuser: "try" });
        }
    } catch (error) {
        console.error(error);
    }
}

export const extractDbusValue = (value) => Array.isArray(value) ? value[0] : value;

/**
 * Checks if a specific Debian package is installed using `dpkg -l`.
 *
 * @param {string} packageName - Name of the package to check.
 * @returns {Promise<boolean>} - Whether the package is installed.
 */
export async function isPackageInstalled (packageName) {
    try {
        await cockpit.spawn([
            "bash",
            "-c",
            `dpkg -l | grep -w '^ii  ${packageName}' `
        ]);
        return true;
    } catch (error) {
        return false;
    }
}

/**
 * Checks whether the firewall package is installed on the system.
 *
 * @return {Promise<boolean>} Returns true if the firewall package "firewalld" is installed, otherwise false.
 */
export function isFirewallInstalled () {
    return isPackageInstalled("firewalld");
}

/**
 * Retrieves a list of country codes by reading and processing the ISO 3166 country codes file.
 * The result excludes comments, blank lines, and undefined entries.
 *
 * @return {Promise<string[]>} A promise that resolves to an array of unique country codes.
 * @throws {Error} Throws an error when there is an issue reading or processing the file.
 */
export async function getAllCountryCodes () {
    try {
        const fileContent = await cockpit.file("/usr/share/zoneinfo/iso3166.tab")
            .read();
        const countryCodes = fileContent.split("\n")
            .map(line => {
                const entries = line.split("\t");
                return entries[0] + " " + entries[1];
            })
            .filter(function (entry) {
                return entry !== "" && entry[0] !== "#" && entry.trim() !== "undefined";
            });
        return [...new Set(countryCodes.slice(countryCodes.indexOf("#code") + 1))];
    } catch (error) {
        console.error("An error occurred while loading wlan country codes.", error);
        throw new Error("An error occurred while loading country codes: " + error);
    }
}

/**
 * Retrieves the current Wi-Fi country code using the `raspi-config` command.
 *
 * This method executes a system-level command to fetch the Wi-Fi country code
 * configured in the Raspberry Pi device. If the command fails or encounters an
 * error, it logs the issue and returns an empty string as a fallback.
 *
 * @return {Promise<string>} A promise that resolves to the country code as a string,
 *                          or an empty string if the retrieval fails.
 */
export async function getCurrentCountryCode () {
    try {
        const code = await cockpit.spawn(["raspi-config", "nonint", "get_wifi_country"], { superuser: "required" });
        return code;
    } catch (e) {
        console.log("raspi-config exited with code 1; setting country code to an empty string.");
        return "";
    }
}

/**
 * Sets a new Wi-Fi country code using the given country code.
 * This function uses a system-specific command to configure the Wi-Fi country settings.
 *
 * @param {string} newCode - The new country code to set, in the format required by the system.
 * @return {Promise<void>} Resolves if the country code is successfully set. Rejects and throws an error if the operation fails.
 */
export async function setNewCountryCode (newCode) {
    try {
        await cockpit.spawn(["raspi-config", "nonint", "do_wifi_country", newCode], { superuser: "require" });
    } catch (e) {
        console.error("Could not set the new wlan country code.", e);
        throw new Error("Could not set the new wlan country code.");
    }
}

/**
 * Detects whether a factory reset has been performed on the device
 * by checking the existence of a specific file.
 *
 * @return {Promise<boolean>} A promise that resolves to true if the file indicating a factory reset exists, otherwise false.
 */
export async function detectFactoryReset () {
    return fileExists("/etc/revpi/factory-reset");
}

/**
 * Detects the presence of a HAT (Hardware Attached on Top) EEPROM by checking
 * if the custom HAT configuration file exists in the device's file system.
 *
 * @return {Promise<boolean>} A promise that resolves to true if the HAT EEPROM file exists,
 *                            otherwise resolves to false.
 */
export async function detectHatEeprom () {
    return fileExists("/proc/device-tree/hat/custom_1");
}

export async function doFactoryReset ({ overlay, serial, mac } = {}) {
    // always check EEPROM at runtime
    const hasHatEeprom = await detectHatEeprom();

    let cmd;
    if (hasHatEeprom) {
        cmd = ["revpi-factory-reset"];
    } else {
        if (!overlay || !serial || !mac) {
            throw new Error("overlay, serial, and mac are required when no HAT EEPROM is present");
        }
        cmd = ["revpi-factory-reset", overlay, serial, mac];
    }

    const stdout = await cockpit.spawn(cmd, { superuser: "require", err: "message" });

    const pwMatch = stdout.match(/^\s*Password:\s*([^\s]+)\s*$/mi);
    const hnMatch = stdout.match(/^\s*Hostname:\s*([^\s]+)\s*$/mi);
    return {
        password: pwMatch?.[1],
        hostname: hnMatch?.[1],
        stdout
    };
}

/**
 * Invokes a system reboot command by executing it as a superuser.
 * This function utilizes the `cockpit.spawn` method to issue the reboot command.
 *
 * @return {Promise<void>} A promise that resolves when the reboot command is successfully initiated.
 */
export async function reboot () {
    await cockpit.spawn(["reboot"], { superuser: "require" });
}

/* ============================================================================
 * Internal utilities (private)
 * ========================================================================== */

/** Normalize a string output from spawn into non-empty trimmed lines. */
function toLines (out) {
    return String(out || "").split("\n").map(s => s.trim()).filter(Boolean);
}

/** Normalize locale like `de_DE.UTF-8@euro` → `de_DE.UTF-8` (keeps charset). */
function normalizeLocale (loc) {
    if (!loc) return "";
    return String(loc).replace(/@.*$/, "");
}

/* ============================================================================
 * Timezone: set, get current, list, region/zone helpers
 * ========================================================================== */

/**
 * Sets the system timezone to the specified timezone.
 * @param {string} tz - e.g. "Europe/Berlin"
 */
export async function setTimezone (tz) {
    const val = String(tz || "").trim();
    if (!val) throw new Error("Timezone is empty");
    await cockpit.spawn(
        ["/usr/bin/raspi-config", "nonint", "do_change_timezone", val],
        { superuser: "require", err: "out" }
    );
}

/** List all available timezones via `timedatectl list-timezones`. */
export async function listTimezones () {
    const out = await cockpit.spawn(
        ["/usr/bin/timedatectl", "list-timezones"],
        { superuser: "require", err: "out" }
    );
    return toLines(out);
}

/** Get current timezone via `timedatectl show -p Timezone --value`. */
export async function currentTimezone () {
    const out = await cockpit.spawn(
        ["/usr/bin/timedatectl", "show", "-p", "Timezone", "--value"],
        { superuser: "require", err: "out" }
    );
    return String(out || "").trim();
}

/** Fixed order of buckets as in your UI */
export const TIMEZONE_BUCKET_ORDER = [
    "Africa",
    "Americas",
    "Antarctica",
    "Arctic Ocean",
    "Asia",
    "Atlantic Ocean",
    "Australia",
    "Europe",
    "Indian Ocean",
    "Pacific Ocean",
    "US",
    "None of the above"
];

/** Map first TZ path segment → UI bucket */
const FIRST_SEGMENT_TO_BUCKET = new Map([
    // continents
    ["Africa", "Africa"],
    ["America", "Americas"],
    ["Antarctica", "Antarctica"],
    ["Arctic", "Arctic Ocean"],
    ["Asia", "Asia"],
    ["Atlantic", "Atlantic Ocean"],
    ["Australia", "Australia"],
    ["Europe", "Europe"],
    ["Indian", "Indian Ocean"],
    ["Pacific", "Pacific Ocean"],

    // special sets that should appear under “Americas” in the UI
    ["Canada", "Americas"],
    ["Mexico", "Americas"],
    ["Brazil", "Americas"],
    ["Chile", "Americas"],
    ["Cuba", "Americas"],
    ["Jamaica", "Americas"],

    // keep US as its own bucket
    ["US", "US"]
]);

/** Return the UI bucket name for a timezone id (e.g. "Europe/Berlin"). */
export function bucketForTimezone (tzId) {
    const first = String(tzId).split("/")[0] || "";
    return FIRST_SEGMENT_TO_BUCKET.get(first) || "None of the above";
}

/**
 * Group a list of timezone IDs into UI buckets.
 * @param {string[]} tzList - e.g. ["Europe/Berlin", "US/Eastern", "America/Sao_Paulo", ...]
 * @return {Record<string, string[]>} object with bucket → array of tz ids (sorted)
 */
export function groupTimezonesIntoBuckets (tzList) {
    const result = Object.fromEntries(TIMEZONE_BUCKET_ORDER.map(name => [name, []]));

    for (const tz of tzList || []) {
        const bucket = bucketForTimezone(tz);
        (result[bucket] ||= []).push(tz);
    }

    // sort each bucket’s contents for stable UI
    for (const name of Object.keys(result)) {
        result[name].sort((a, b) => a.localeCompare(b));
    }
    return result;
}

/**
 * Convenience: return an array ready for rendering, in the fixed bucket order.
 * Each item: { name: string, zones: string[] }
 */
export function getTimezoneBuckets (tzList) {
    const grouped = groupTimezonesIntoBuckets(tzList);
    return TIMEZONE_BUCKET_ORDER
        .map(name => ({ name, zones: grouped[name] }))
        .filter(item => item.zones.length > 0 || item.name === "None of the above"); // keep empty only if you want
}

/* ============================================================================
 * Locale: set, get current, list supported
 * ========================================================================== */

/**
 * Sets the system locale.
 * @param {string} locale - e.g. "de_DE.UTF-8"
 */
export async function setLocale (locale) {
    const val = String(locale || "").trim();
    if (!val) throw new Error("Locale is empty");
    await cockpit.spawn(
        ["/usr/bin/raspi-config", "nonint", "do_change_locale", val],
        { superuser: "require", err: "message" }
    );
}

/**
 * List supported locales from /usr/share/i18n/SUPPORTED.
 * @param {boolean} [utf8Only=true]
 */
export async function listSupportedLocales (utf8Only = true) {
    // Each line like: "de_DE.UTF-8 UTF-8"
    const out = await cockpit.spawn(
        ["/usr/bin/awk", "!/^#/ {print $1 \"|\" $2}", "/usr/share/i18n/SUPPORTED"],
        { superuser: "require", err: "out" }
    );

    const entries = toLines(out).map(line => {
        const [locale, charset] = line.split("|");
        return { locale, charset };
    });

    return entries
        .filter(e => (utf8Only ? /utf-?8/i.test(e.charset) : true))
        .map(e => e.locale)
        .filter(Boolean);
}

/**
 * Get current system locale from `localectl`.
 * Parses `System Locale: LANG=...` line.
 */
export async function currentLocale () {
    const out = await cockpit.spawn(
        ["/usr/bin/localectl"],
        { superuser: "require", err: "out" }
    );
    const systemLocale = String(out || "").match(/System Locale:\s*([^\n]+)/);
    if (!systemLocale) return "";
    const localectlLine = systemLocale[1].trim();
    const localeMatch = localectlLine.match(/LANG=([^,\s]+)/);
    return normalizeLocale(localeMatch ? localeMatch[1] : "");
}

/* ============================================================================
 * Keyboard: set, get current, list layouts
 * ========================================================================== */

/**
 * Update keyboard layout (e.g., "de", "us", "gb").
 * @param {string} layout
 */
export async function setKeyboardLayout (layout) {
    const val = String(layout || "").trim();
    if (!val) throw new Error("Keyboard layout is empty");
    await cockpit.spawn(
        ["/usr/bin/raspi-config", "nonint", "do_configure_keyboard", val],
        { superuser: "require", err: "out" }
    );
}

/**
 * Get current keyboard layout from `localectl` (X11 Layout).
 */
export async function getCurrentKeyboardLayout () {
    const out = await cockpit.spawn(
        ["/usr/bin/localectl"],
        { superuser: "require", err: "out" }
    );
    const lines = toLines(out);
    const layoutLine = lines.find(l => l.startsWith("X11 Layout:"));
    const match = layoutLine?.match(/X11 Layout:\s*([a-zA-Z0-9_-]+)/);
    return match ? match[1] : "";
}

/**
 * List all X11 keymap layouts via `localectl list-x11-keymap-layouts`.
 * Returns an array of layout codes (e.g., ["us","de","gb",...]).
 */
export async function getAllKeyboardLayouts () {
    const out = await cockpit.spawn(
        ["/usr/bin/localectl", "list-x11-keymap-layouts"],
        { superuser: "require", err: "out" }
    );
    return toLines(out);
}
