# -*- coding: utf-8 -*-
"""Test OPC UA Server."""
__author__ = "Sven Sager"
__copyright__ = "Copyright (C) 2023 KUNBUS GmbH"
__license__ = "GPLv2"

import unittest
from collections import namedtuple
from logging import getLogger
from os.path import abspath, dirname, join
from socket import getfqdn
from tempfile import NamedTemporaryFile
from time import sleep, time
from typing import Optional

from asyncua import Node
from asyncua.crypto.security_policies import SecurityPolicyBasic256Sha256
from asyncua.sync import Client
from asyncua.ua import VariantType
from asyncua.ua.uaerrors import BadUserAccessDenied
from revpimodio2 import RevPiModIO, RevPiModIOSelected
from revpimodio2.device import Device, Virtual
from revpimodio2.io import IOBase
from revpimodio2.modio import DevSelect

from opcua_revpi_server import ua_server
from opcua_revpi_server.ua_server import SharedIos, UaServer

log = getLogger(__name__)
BASE_PATH = dirname(abspath(__file__))
SERVER_JOIN_TIME = 10


class OpcUaServerBasics(unittest.TestCase):
    """
    Test functions for the instantiation of the UaServer class - no start.

    Various piCtory configurations for instantiation are tested. In the case
    of valid configurations, the recognized values are checked. In the case of
    incorrect instantiations, the error handling of the class is tested.
    """

    def setUp(self) -> None:
        # Prepare empty process image
        self.procimg = NamedTemporaryFile("rb+", 0)
        self.procimg.write(b"\x00" * 4096)

    def tearDown(self) -> None:
        # Close process image file, this will delete it as well
        self.procimg.close()

    def test_valid_pictory(self):
        """
        Test a valid piCtory configuration.

        Diese Konfiguration hat auf Position 64 einen virtuellen OPC UA Server.
        Die Konfiguration besteht aus einem AIO, DIO, Connect 3, OPC UA Server.
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24585",)),
            configrsc=join(BASE_PATH, "pictory_configs/aio_dio_connect_opc.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)
        ua_server.UaServer(rpi.device[64])

    def test_get_properies(self):
        """Test a valid piCtory configuration from test_valid_pictory."""
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24585",)),
            configrsc=join(BASE_PATH, "pictory_configs/aio_dio_connect_opc.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)

        server = ua_server.UaServer(rpi.device[64])
        self.assertTrue(server.host_name)
        self.assertEqual(server.host_name, getfqdn())
        self.assertEqual(server.host, "localhost")
        self.assertEqual(server.port, 4840)

        server = ua_server.UaServer(rpi.device[64], "127.0.0.1")
        self.assertEqual(server.host, "127.0.0.1")

    def test_invalid_device_id(self):
        """Configuration has an unknown device ID.

        The OPC UA Server device has an unknown device ID. We set it to
        '20230929_0_0' instead of '20230929_1_0'.
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24585",)),
            configrsc=join(BASE_PATH, "pictory_configs/invalid_deviceid.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)

        with self.assertRaisesRegex(
            RuntimeError,
            "The device version 'device_OPCUARevPiServer_20230929_0_0_001' "
            "of opc ua server is not supported",
        ):
            ua_server.UaServer(rpi.device[64])

    def test_shared_ios_values(self):
        """
        Check the enum values between piCtory and the ua server.

        The piCtory configuration has two OPC UA server devices. Position 64
        exports all existing IOs, position 65 exports only the exported IOs.

        Position 64, 65 = OPCUARevPiServer_20230929_1_0
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24585",)),
            configrsc=join(BASE_PATH, "pictory_configs/shared_io_settings.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 2)

        server = ua_server.UaServer(rpi.device[64])
        self.assertIs(server._shared_ios, SharedIos.ALL)

        server = ua_server.UaServer(rpi.device[65])
        self.assertIs(server._shared_ios, SharedIos.EXPORTED)

    def test_synchronization_rate_change(self):
        """Test getter and setter to change synchronization rate."""
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24585",)),
            configrsc=join(BASE_PATH, "pictory_configs/aio_dio_connect_opc.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)
        server = ua_server.UaServer(rpi.device[64])

        # Default is 10 Hz
        self.assertEqual(server.synchronization_rate, 10)
        with self.assertRaisesRegex(ValueError, r"Synchronization rate must be between 1 and 50"):
            server.synchronization_rate = 51
        with self.assertRaisesRegex(ValueError, r"Synchronization rate must be between 1 and 50"):
            server.synchronization_rate = 0.9

        # The property will calculate the wait time for asyncio main cycle
        server.synchronization_rate = 50
        server._synchronization_wait_time = 0.02
        server.synchronization_rate = 2
        server._synchronization_wait_time = 0.5


class OpcUaServerServe(unittest.TestCase):
    """
    This test case starts the OPC UA server and works with the connection.

    The server is started with valid piCtory configurations and the values
    are exchanged via an OPC UA client.
    """

    def setUp(self) -> None:
        # Prepare empty process image
        self.procimg = NamedTemporaryFile("rb+", 0)
        self.procimg.write(b"\x00" * 4096)

        # Tests uses this for opc ua server to stop it in case of failure
        self.test_server = None

    def tearDown(self) -> None:
        # Stop a running opc ua server in case of failure
        if self.test_server and self.test_server.is_alive():
            log.warning("Teardown function of test is shutting down a running OPC UA server")
            self.test_server.stop()
            self.test_server.join(SERVER_JOIN_TIME)
            if self.test_server.is_alive():
                log.error("TEARDOWN ERROR: OPC UA SERVER STILL RUNNING")

        # Close process image file, this will delete it as well
        self.procimg.close()

    def test_start_stop_server(self):
        """
        Start a server without encryption and check values.

        Test values are written to the process image via RevPiModIO. This
        simulates an independent application that runs on the Revolution Pi.
        Via this process image, the OPC UA server accesses the values and
        makes them available to the client. This picks up the value and
        compares it with the set value.
        """

        # Use a full RevPiModIO to simulate values for the client to read
        rpi = RevPiModIO(
            configrsc=join(BASE_PATH, "pictory_configs/aio_dio_connect_opc.rsc"),
            procimg=self.procimg.name,
            shared_procimg=True,
        )

        # Change the defaultvalue of MEM Replace_IO_file to our test conf
        rpi.io.Replace_IO_file._defaultvalue = join(
            BASE_PATH, "replace_io_files/empty.conf"
        ).encode()

        # The opcua virtual device is on position 64 in pictory configuration
        self.test_server = create_opcua_server(rpi.device[64])
        self.assertIsNotNone(self.test_server)

        # Create a client to connect to server
        with Client(url="opc.tcp://localhost:4840/opcua-revpi-server") as client:
            uri = f"http://{self.test_server.host_name}/UA/virtualopcuaserver"
            idx = client.get_namespace_index(uri)

            # Query the opc ua object from server
            var = client.nodes.root.get_child(
                ["0:Objects", f"{idx}:AnalogesDevice", f"{idx}:OutputValue_1"],
            )

            for i in (100, 26, 0):
                # Set a value in process image, which the opc ua server uses
                rpi.io.OutputValue_1.value = i
                rpi.writeprocimg()
                sleep(0.1)

                # Request the changed value over opc ua server
                value = var.read_value()

                # Compare set value and received value from server
                self.assertEqual(i, value)

        # Send shut down server request
        self.test_server.stop()

        # Wait for server to shut down
        self.test_server.join(SERVER_JOIN_TIME)
        self.assertFalse(self.test_server.is_alive())

    def test_start_stop_server_tls(self):
        """
        Start a server with encryption and check values.


        todo: Change description
        Test values are written to the process image via RevPiModIO. This
        simulates an independent application that runs on the Revolution Pi.
        Via this process image, the OPC UA server accesses the values and
        makes them available to the client. This picks up the value and
        compares it with the set value.
        """

        # Load a piCtory configuration with enabled TLS encryption
        rpi = RevPiModIO(
            configrsc=join(BASE_PATH, "pictory_configs/aio_dio_connect_opctls.rsc"),
            procimg=self.procimg.name,
            shared_procimg=True,
        )

        # Change the defaultvalue of MEM TLS_certificate_path to our test certificate
        rpi.io.TLS_certificate_path._defaultvalue = join(
            BASE_PATH,
            "tls/localhost_opcua_server.pem",
        ).encode()
        # Change the defaultvalue of MEM TLS_Key_path to our test private key
        rpi.io.TLS_Key_path._defaultvalue = join(
            BASE_PATH,
            "tls/localhost_opcua_server_key.pem",
        ).encode()

        # The opcua virtual device is on position 64 in pictory configuration
        self.test_server = create_opcua_server(rpi.device[64])
        self.assertIsNotNone(self.test_server)

        # Create a client to connect to server
        client = Client(url="opc.tcp://localhost:4843/opcua-revpi-server")
        client.application_uri = "urn:localhost:opcua-testing"
        client.set_security(
            SecurityPolicyBasic256Sha256,
            certificate=join(BASE_PATH, "tls/localhost_opcua_server.pem"),
            private_key=join(BASE_PATH, "tls/localhost_opcua_server_key.pem"),
        )
        with client:
            uri = f"http://{self.test_server.host_name}/UA/virtualopcuaserver"
            idx = client.get_namespace_index(uri)

            # Query the opc ua object from server
            var = client.nodes.root.get_child(
                ["0:Objects", f"{idx}:AnalogesDevice", f"{idx}:OutputValue_1"],
            )

            for i in (100, 26, 0):
                # Set a value in process image, which the opc ua server uses
                rpi.io.OutputValue_1.value = i
                rpi.writeprocimg()
                sleep(0.1)

                # Request the changed value over opc ua server
                value = var.read_value()

                # Compare set value and received value from server
                self.assertEqual(i, value)

        # Send shut down server request
        self.test_server.stop()

        # Wait for server to shut down
        self.test_server.join(SERVER_JOIN_TIME)
        self.assertFalse(self.test_server.is_alive())

    def test_serve_replaced_ios(self):
        """
        Start a server without encryption and test replace ios.

        The RevPiModIO2 library supports the replacement of inputs and outputs
        via a "replace_io" configuration. There, several bytes or WORDs are
        combined into new data types. In this test, the OPC UA server is loaded
        with a replace io configuration, which creates a 4 byte unsigned
        integer with the name 'opc_ua_test_output' from two WORDs. This must be
        available via OPC UA.
        """

        replace_io_file = join(BASE_PATH, "replace_io_files/connect_opc_virt.conf")

        # Use a full RevPiModIO to simulate values for the client to read
        rpi = RevPiModIO(
            replace_io_file=replace_io_file,
            configrsc=join(BASE_PATH, "pictory_configs/connect_opc_virt.rsc"),
            procimg=self.procimg.name,
            shared_procimg=True,
        )

        # Pick the opc ua server device from pictory configuraion
        virtual_opcua_server_device = rpi.device[64]

        # Create an opc ua server without existing file for MEM Replace_IO_file
        with self.assertRaisesRegex(
            RuntimeError,
            r"could not read/parse file '/does/not/exist/replace_ios.conf'",
        ):
            # This is an execution raised by RevPiModIO2
            ua_server.UaServer(virtual_opcua_server_device)

        # Change the defaultvalue of MEM Replace_IO_file to our test configuraiton
        rpi.io.Replace_IO_file._defaultvalue = replace_io_file.encode()

        # The opcua virtual device is on position 64 in pictory configuration
        self.test_server = create_opcua_server(rpi.device[64])
        self.assertIsNotNone(self.test_server)

        # Create a client to connect to server
        with Client(url="opc.tcp://localhost:4840/opcua-revpi-server") as client:
            uri = f"http://{self.test_server.host_name}/UA/virtualopcuaserver"
            idx = client.get_namespace_index(uri)

            # Query the opc ua object from server
            var = client.nodes.root.get_child(
                ["0:Objects", f"{idx}:virt01", f"{idx}:opc_ua_test_output"],
            )

            # Values that are larger than the original ones of each output are used.
            for i in (0xFFFFFFFF, 0x00FF0000, 0x00000000):
                # Set a value in process image, which the opc ua server uses
                rpi.io.opc_ua_test_output.value = i
                rpi.writeprocimg()
                sleep(0.1)

                # Request the changed value over opc ua server
                value = var.read_value()

                # Compare set value and received value from server
                self.assertEqual(i, value)

        # Send shut down server request
        self.test_server.stop()

        # Wait for server to shut down
        self.test_server.join(SERVER_JOIN_TIME)
        self.assertFalse(self.test_server.is_alive())


class OpcUaServerInternal(unittest.TestCase):
    """
    This tests work with integrated server functions.

    This test case generates a ModIO object and two OPC UA servers for all
    tests. One server has set the output mode 'read only' and the second
    'read and write'.
    Clients are generated, each of which is connected to a server.
    """

    def setUp(self) -> None:
        """Create two opc ua server and connected clients for each test."""
        # Prepare empty process image
        self.procimg = NamedTemporaryFile("rb+", 0)
        self.procimg.write(b"\x00" * 4096)

        # Use a full RevPiModIO to simulate values for the client to read
        self.rpi = RevPiModIO(
            configrsc=join(BASE_PATH, "pictory_configs/server_internal_setup.rsc"),
            procimg=self.procimg.name,
            replace_io_file=join(BASE_PATH, "replace_io_files/server_internal_setup.conf"),
            shared_procimg=True,
        )
        # Change the defaultvalue of MEM Replace_IO_file to our test conf
        self.rpi.io.Replace_IO_file._defaultvalue = join(
            BASE_PATH, "replace_io_files/server_internal_setup.conf"
        ).encode()
        self.rpi.io.Replace_IO_file_i06._defaultvalue = join(
            BASE_PATH, "replace_io_files/server_internal_setup.conf"
        ).encode()

        # Start the servers and wait, retry the start until the bind was successful
        self.server_ro = create_opcua_server(self.rpi.device[64])
        self.assertIsNotNone(self.server_ro)
        self.server_rw = create_opcua_server(self.rpi.device[65])
        self.assertIsNotNone(self.server_rw)

        self.client_ro = Client(url="opc.tcp://localhost:4840/opcua-revpi-server")
        self.client_rw = Client(url="opc.tcp://localhost:4841/opcua-revpi-server")
        self.client_ro.connect()
        self.client_rw.connect()

        uri = f"http://{self.server_ro.host_name}/UA/virtualopcuaserver"
        self.idx = self.client_ro.get_namespace_index(uri)

    def tearDown(self) -> None:
        """Disconnect clients and shut down servers."""
        # Disconnect client form OPC UA
        self.client_ro.disconnect()
        self.client_rw.disconnect()

        # Stop a running opc ua server
        self.server_ro.stop()
        self.server_rw.stop()
        self.server_ro.join(SERVER_JOIN_TIME)
        self.server_rw.join(SERVER_JOIN_TIME)
        if self.server_ro.is_alive():
            log.error("TEAR-DOWN ERROR: OPC UA (RO) SERVER STILL RUNNING")
        if self.server_rw.is_alive():
            log.error("TEAR-DOWN ERROR: OPC UA (RW) SERVER STILL RUNNING")

        # Close process image file, this will delete it as well
        self.procimg.close()

    def test_ro_rw(self):
        """Check set Output_mode in PiCtory configuration ro/rw."""
        self.assertTrue(self.server_ro.is_alive())
        self.assertTrue(self.server_rw.is_alive())

        for dev in self.rpi.device:  # type: Device
            for io in dev.get_outputs():  # type: IOBase
                ua_object = ["0:Objects", f"{self.idx}:{dev.name}", f"{self.idx}:{io.name}"]
                var_ro = self.client_ro.nodes.root.get_child(ua_object)  # type: Node
                var_rw = self.client_rw.nodes.root.get_child(ua_object)  # type: Node

                # Write output on a OPC UA server with Output_mode=Read only
                with self.assertRaisesRegex(
                    BadUserAccessDenied,
                    r"User does not have permission to perform the requested operation.",
                ):
                    var_ro.write_value(1, var_ro.read_data_type_as_variant_type())

                # Write output on a OPC UA server with Output_mode=Read and write
                self.assertEqual(int(io.value), int(var_rw.read_value()))
                var_rw.write_value(1, var_rw.read_data_type_as_variant_type())

            sleep(0.2)
            self.rpi.readprocimg()
            for io in dev.get_outputs():  # type: IOBase
                self.assertEqual(int(io.value), 1)

    def test_data_types(self):
        """Check OPC UA datatypes of IOs."""
        Check = namedtuple("Check", ["device_name", "io_name", "datatype"])

        # Datatypes of virt01 are defined as replace_ios
        lst_io_mapping = [
            Check("virt02", "OutBit_1", VariantType.Boolean),
            Check("virt02", "OutByte_10", VariantType.Byte),
            Check("virt01", "my_sbyte", VariantType.SByte),
            Check("virt01", "my_int16", VariantType.Int16),
            Check("virt02", "OutWord_1", VariantType.UInt16),
            Check("virt01", "my_int32", VariantType.Int32),
            Check("virt02", "OutDWord_1", VariantType.UInt32),
            Check("virt01", "my_int64", VariantType.Int64),
            Check("virt01", "my_uint64", VariantType.UInt64),
            Check("virt01", "my_float", VariantType.Float),
            Check("virt01", "my_double", VariantType.Double),
            Check("virt01", "my_bytearray", VariantType.ByteString),
        ]

        for check in lst_io_mapping:
            ua_object = [
                "0:Objects",
                f"{self.idx}:{check.device_name}",
                f"{self.idx}:{check.io_name}",
            ]
            var_ro = self.client_ro.nodes.root.get_child(ua_object)  # type: Node
            self.assertIs(var_ro.read_data_type_as_variant_type(), check.datatype)


def create_opcua_server(virtual_device: Virtual, timeout: int = 300) -> Optional[UaServer]:
    """
    Creates functional OPC UA servers within the timeout.

    On different test devices, it can happen that the rapid generation of the
    same socket connections leads to errors if the operating system does not
    destroy the sockets quickly enough.

    If the timeout is exceeded, None will be returned. This should be checked
    in tests with 'assertIsNotNone(...)'.

    :param virtual_device: Virtual device with OPC UA server configuration
    :param timeout: After the timeout, return None
    :returns: Fully started and usable OPC UA Server or None
    """
    if not 1 <= timeout <= 300:
        raise ValueError("timeout must be in range 1 - 300 seconds")

    timer = time() + timeout
    while time() < timer:
        created_server = ua_server.UaServer(virtual_device)
        created_server.start()
        if created_server.startup_complete.wait(timeout):
            if created_server.error_code == 0:
                return created_server
        sleep(1.0)

    log.error("Could not create an opc ua server for testing")
