# -*- coding: utf-8 -*-
"""OPC UA server for Revolution Pi."""
__author__ = "Sven Sager"
__copyright__ = "Copyright (C) 2023 KUNBUS GmbH"
__license__ = "GPLv2"

import asyncio
from enum import IntEnum
from logging import getLogger
from socket import getfqdn
from threading import Event, Thread

from asyncua import Node, Server, ua
from asyncua.ua import ApplicationType, VariantType
from revpimodio2 import OUT, RevPiModIO
from revpimodio2.device import Virtual

from . import __about__
from .helper import get_ua_datatype

log = getLogger(__name__)


class SharedIos(IntEnum):
    """Translate enum values form pictory virtual device."""

    NONE = 0
    OWN = 1
    EXPORTED = 2
    ALL = 3


class OpcUaVariableWrapper:
    """
    Manages an OPC UA variable and its last value.

    Values of OPC UA variables are read and set transparently via this
    wrapper. When setting, the new value is buffered and compared with the
    currently set value in the OPC UA server when 'has_new_value' is called.
    This makes it possible to check whether values in the OPC UA server differ
    from the last set values.
    """

    def __init__(self, opcua_var: Node, opcua_datatype: VariantType):
        self._last_value = None
        self.opcua_type = opcua_datatype
        self.opcua_var = opcua_var

    async def has_new_value(self) -> bool:
        """
        Check whether a new value has been set by the client.

        If a value has never been set with set_value(), this function always
        returns False.
        """
        if self._last_value is None:
            return False
        else:
            return self._last_value != await self.opcua_var.read_value()

    async def read_value(self):
        return await self.opcua_var.read_value()

    async def set_value(self, value) -> None:
        """Set value and save it as reference for has_new_value function."""
        self._last_value = value
        await self.opcua_var.set_value(value, self.opcua_type)


class UaServer(Thread):
    def __init__(self, virtual_device: Virtual, host="localhost"):
        """
        OPC UA server with configuration in a virtual device.

        The default synchronization rate between process image and OPC UA
        values is 10 hearts. It can be queried and adjusted via
        '.synchronization_rate' before the start or during the runtime.

        :param virtual_device: Use this virtual device for configuration values
        :param host: Bind-Address of OPC UA server
        """
        super().__init__()
        self._error_code = 0
        self._evt_exit = Event()
        self._synchronization_wait_time = 0.1
        self.startup_complete = Event()

        # Remove the incrementing number of id to get the pure version string
        vdev_id = virtual_device.id.rsplit("_", maxsplit=1)[0]

        # This supports multiple versions of the opc ua virtual device
        if vdev_id == "device_OPCUARevPiServer_20230929_1_0":
            log.debug("PiCtory module: OPCUARevPiServer_20230929_1_0")
            self._port = virtual_device[0].defaultvalue
            self._shared_ios = SharedIos(virtual_device[1].defaultvalue)
            self._allow_write = bool(virtual_device[2].defaultvalue)
            self._replace_io = virtual_device[3].defaultvalue or None
            self._use_tls_encryption = bool(virtual_device[4].defaultvalue)
            self._tls_certificate_path = virtual_device[5].defaultvalue
            self._tls_key_path = virtual_device[6].defaultvalue
        else:
            raise RuntimeError(
                f"The device version '{virtual_device.id}' of opc ua server is not supported"
            )

        # Read content of files to be able to drop permissions for server process
        self._tls_certificate = b""
        self._tls_key = b""
        if self._use_tls_encryption:
            with open(self._tls_certificate_path, "rb") as fh:
                self._tls_certificate = fh.read()
                log.info(f"Using certificate '{self._tls_certificate_path}'")
            with open(self._tls_key_path, "rb") as fh:
                log.info(f"Using private key '{self._tls_key_path}'")
                self._tls_key = fh.read()

        self._bind_host = host
        self._host_name = getfqdn()
        self._modio = RevPiModIO(
            procimg=virtual_device._modio.procimg,
            configrsc=virtual_device._modio.configrsc,
            replace_io_file=self._replace_io,
            shared_procimg=True,
        )

    async def __ua_srv(self):
        """This is the async io entry point of the opc ua server process."""
        ua_srv = Server()

        # This values can be set before the init call. They will be used in the .init function.
        ua_srv.application_type = ApplicationType.Server
        ua_srv.manufacturer_name = "KUNBUS GmbH"
        ua_srv.name = "opcua-revpi-server"
        ua_srv.product_uri = "urn:revolutionpi.com:opcua"

        await ua_srv.init()

        # The .init call calls set_build_info without software_version, so we call it again
        await ua_srv.set_build_info(
            ua_srv.product_uri,
            ua_srv.manufacturer_name,
            ua_srv.name,
            __about__.__version__,
            __about__.build_number,
            __about__.build_datetime,
        )

        # Now set values, which needs a full initialized server instance
        ua_srv.set_server_name("Revolution Pi OPC UA server")
        ua_srv.set_endpoint(f"opc.tcp://0.0.0.0:{self._port}/opcua-revpi-server")
        await ua_srv.set_application_uri(f"urn:{self._host_name}:opcua-revpi-server")

        if self._use_tls_encryption:
            # TLS is enabled by user, this will the policy and load the certificate
            lst_security_policy = [
                ua.SecurityPolicyType.Basic256Sha256_SignAndEncrypt,
                ua.SecurityPolicyType.Basic256Sha256_Sign,
            ]
            await ua_srv.load_certificate(self._tls_certificate, format="pem")
            await ua_srv.load_private_key(self._tls_key, format="pem")
        else:
            # No encryption is enabled by user, so the only policy which works is NoSecurity
            lst_security_policy = [
                ua.SecurityPolicyType.NoSecurity,
            ]
        ua_srv.set_security_policy(lst_security_policy)

        # Now we set up a free namespace inside the server for our IOs
        uri = f"http://{self._host_name}/UA/virtualopcuaserver"
        idx = await ua_srv.register_namespace(uri)

        # Create all devices
        lst_modio_ua_mappings = []
        for dev in self._modio.device:
            # Create a device in OPC UA to add IOs to
            ua_dev = await ua_srv.nodes.objects.add_object(idx, dev.name)

            for io in dev.get_inputs() + dev.get_outputs():
                if self._shared_ios is SharedIos.EXPORTED and not io.export:
                    # Add IOs with export flag only
                    continue

                ua_datatype = get_ua_datatype(io)
                ua_var = await ua_dev.add_variable(idx, io.name, io.value, ua_datatype)

                if io.type is OUT and self._allow_write:
                    await ua_var.set_writable()

                lst_modio_ua_mappings.append(
                    # Map ModIO and UA IOs in a tuple to update the values in mainloop
                    (io, OpcUaVariableWrapper(ua_var, ua_datatype)),
                )

        # Start server object
        await ua_srv.start()
        log.debug(f"Awaited OPC UA server start {self._bind_host}:{self._port}")
        self.startup_complete.set()

        # Async mainloop
        while not self._evt_exit.is_set():
            # In NONE mode, no RevPi IOs are shared with OPC UA - users can do their own OPC UA stuff
            if self._shared_ios is SharedIos.NONE:
                await asyncio.sleep(self._synchronization_wait_time)
                continue

            # Update all ModIo values to OPC UA variables
            self._modio.readprocimg()

            for io, var in lst_modio_ua_mappings:
                if self._allow_write and io.type is OUT and await var.has_new_value():
                    # Write only changed opc ua values to do not override the complete process image
                    try:
                        io.value = await var.read_value()
                    except Exception as e:
                        # The old ModIO value will be set to OPC UA at the end of the for loop
                        log.error(f"Could not set new value to process image: '{e}'")

                # Set all ModIO values to OPC UA variables, which will enable has_new_value check of var object
                await var.set_value(io.value)

            if self._allow_write:
                self._modio.writeprocimg()

            await asyncio.sleep(self._synchronization_wait_time)

        log.debug(f"Await stopping OPC UA server {self._bind_host}:{self._port}")
        await ua_srv.stop()
        log.debug(f"Awaited OPC UA server stop {self._bind_host}:{self._port}")

    def run(self) -> None:
        """Start the async opc ua server."""
        log.info(f"Start OPC UA server {self._bind_host}:{self._port}")
        try:
            asyncio.run(self.__ua_srv())
            log.info(f"Stopped OPC UA server {self._bind_host}:{self._port}")
        except Exception as e:
            log.error(f"OPC UA server crashed: {str(e)}")
            self._error_code = 1
            self.startup_complete.set()

    def stop(self) -> None:
        """
        Request to stop the opc ua server.

        You have to check .is_alive or .join this object to check the end of
        execution.
        """
        log.debug("Enter UaServer.stop()")
        self._evt_exit.set()
        log.debug("Leave UaServer.stop()")

    @property
    def error_code(self) -> int:
        return self._error_code

    @property
    def host(self) -> str:
        """Get bind address of this server instance."""
        return self._bind_host

    @property
    def host_name(self):
        """Get the used hostname for all URIs."""
        return self._host_name

    @property
    def port(self) -> int:
        """Get port of this server instance."""
        return self._port

    @property
    def synchronization_rate(self) -> int:
        """Get synchronization rate of process image and OPC UA values."""
        return int(1 / self._synchronization_wait_time)

    @synchronization_rate.setter
    def synchronization_rate(self, value: int) -> None:
        """Set synchronization rate of process image and OPC UA values."""
        if not 1 <= value <= 50:
            raise ValueError("Synchronization rate must be between 1 and 50")
        self._synchronization_wait_time = 1 / value
