# -*- coding: utf-8 -*-
"""OPC UA server for Revolution Pi."""
__author__ = "Sven Sager"
__copyright__ = "Copyright (C) 2023 KUNBUS GmbH"
__license__ = "GPLv2"

from logging import getLogger
from threading import Event
from time import perf_counter
from typing import List

from revpimodio2 import RevPiModIOSelected
from revpimodio2.errors import DeviceNotFoundError
from revpimodio2.modio import DevSelect

from . import proginit as pi
from .ua_server import UaServer
from .watchdogs import ResetDriverWatchdog

log = getLogger(__name__)


class RevPiUaServerManager:
    """Main program of RevPiUaServerManager class."""

    def __init__(self):
        """Init RevPiUaServerManager class."""
        log.debug("enter RevPiUaServerManager.__init__")

        self._cycle_time = 1.0
        self.do_cycle = Event()
        self._running = True

        # Set class variables
        self.host = "0.0.0.0"
        self.ua_servers = []  # type: List[UaServer]
        self.ua_startup_timeout = 60.0

        log.debug("leave RevPiUaServerManager.__init__")

    def opcua_reload(self) -> int:
        """
        Stop all OPC UA servers, create new ones and start them.

        All currently running OPC US servers will be shut down. The OPC UA
        servers are created and started according to the PiCtory configuration
        with their settings. Afterward, all servers are waited for whether
        they have started or returned an error. The error_code can be used to
        identify the errors.

        The error code is bit-coded. The bits can be used to find out what
        caused an error.
        Error bits, if set, this error occurred.
            Bit 0 = Could not stop all OC UA servers
            Bit 1 = Cloud not load RevPiModIO configuration
            Bit 2 = Could not create one or more OPC UA servers
            Bit 3 = One or more OPC UA servers runs into startup timeout
            Bit 4 = One or more OPC UA server had errors during startup

        :returns: Error code (See description)
        """
        log.debug("enter RevPiUaServerManager.opcua_reload")

        # Shut down any existing opc ua servers
        error_code = self.opcua_stop()

        if not error_code:
            try:
                # Filter for 'OPC UA Server' virtual devices, which have productType 24585
                modio = RevPiModIOSelected(
                    DevSelect(search_key="productType", search_values=("24585",)),
                    procimg=pi.pargs.procimg,
                    configrsc=pi.pargs.configrsc,
                    shared_procimg=True,
                )
            except DeviceNotFoundError:
                # No configured devices in PiCtory is not an error
                log.warning("No 'OPC UA Server' devices configured in PiCtory.")

            except Exception as e:
                log.error(f"Can not load RevPiModIO: {e}")
                error_code |= 2

            else:
                # Start server instances
                for device in modio.device:
                    log.debug(f"Use virtual device on position {device.position}")
                    try:
                        server = UaServer(device, self.host)
                        server.start()
                    except Exception as e:
                        log.error(
                            f"Could not create OPC UA server on position {device.position}: {e}"
                        )
                        error_code |= 4
                    else:
                        self.ua_servers.append(server)
                        log.info(f"Started opc ua server on virtual device {device.position}")

            # Set time until all instances must be started
            max_startup_time = perf_counter() + self.ua_startup_timeout
            for ua_server in self.ua_servers.copy():
                # Calculates the remaining waiting time that is available
                remaining_startup_timeout = max_startup_time - perf_counter()
                if remaining_startup_timeout < 0.0:
                    # This value must never be less than 0, as .wait would throw an exception
                    remaining_startup_timeout = 0.0

                # Wait until the server is able to process client requests.
                if not ua_server.startup_complete.wait(remaining_startup_timeout):
                    log.error(
                        f"OPC UA server start timeout exceeded {ua_server.host}:{ua_server.port}"
                    )
                    ua_server.stop()
                    self.ua_servers.remove(ua_server)
                    error_code |= 8

                # Check whether errors occurred during server startup.
                if ua_server.error_code:
                    log.error(
                        f"OPC UA server error '{ua_server.error_code}' "
                        f"on startup {ua_server.host}:{ua_server.port}"
                    )
                    ua_server.stop()
                    self.ua_servers.remove(ua_server)
                    error_code |= 16

        log.debug("leave RevPiUaServerManager.opcua_reload")
        return error_code

    def opcua_stop(self) -> int:
        """Stop all OPC UA servers."""
        log.debug("enter RevPiUaServerManager.opcua_stop")

        error_code = 0

        log.debug("Stop all opc ua servers")
        for server in self.ua_servers:
            server.stop()
            server.join(5.0)
            if server.is_alive():
                error_code = 1
                log.error(f"Could not stop ua server {server.host}:{server.port}")

        # Remove all destroyed opc ua servers from list. This will be rebuilt on opcua_start.
        self.ua_servers.clear()

        log.debug("leave RevPiUaServerManager.opcua_stop")
        return error_code

    @staticmethod
    def rotate_logfile() -> None:
        """Start a new logfile."""
        log.debug("enter RevPiUaServerManager.rotate_logfile")

        pi.reconfigure_logger()
        log.warning("start new logfile")

        log.debug("leave RevPiUaServerManager.rotate_logfile")

    def start(self) -> int:
        """Blocking mainloop of program."""
        log.debug("enter RevPiUaServerManager.start")
        error_code = 0

        # Start all configured OPC UA servers
        self.opcua_reload()

        # Startup tasks
        driver_reset = ResetDriverWatchdog()

        pi.startup_complete()
        # Go into mainloop of daemon
        while self._running:
            ot = perf_counter()
            self.do_cycle.clear()

            # Main tasks of cycle loop
            if driver_reset.triggered:
                self.opcua_reload()

            dm = divmod(ot - perf_counter(), self._cycle_time)
            # For float the result is (q, a % b), where q is usually math.floor(a / b) but may be 1 less than that.
            if dm[0] == -1.0:  # Cycle time not exceeded
                self.do_cycle.wait(dm[1])
            else:
                log.debug("Cycle time exceeded about {0:.0f} times".format(abs(dm[0])))

        # Cleanup tasks
        driver_reset.stop()
        self.opcua_stop()

        log.debug("leave RevPiUaServerManager.start")
        return error_code

    def stop(self) -> None:
        """Set stop request for mainloop."""
        log.debug("enter RevPiUaServerManager.stop")

        self._running = False
        self.do_cycle.set()

        log.debug("leave RevPiUaServerManager.stop")


def main() -> int:
    # Check all sub commands
    if pi.pargs.tools == "pictory":
        # Execution with sub-command pictory
        from custom_rap_installer import PiCtoryCommand, main as pictory_main
        from os.path import dirname, join

        source_dir = join(dirname(__file__), "rap_installer_source")
        return_code = pictory_main(
            (
                PiCtoryCommand(pi.pargs.pictory_commands)
                if pi.pargs.pictory_commands
                else PiCtoryCommand.INFO
            ),
            source_dir,
            pi.pargs.install_root,
            pi.pargs.remove if "remove" in pi.pargs else False,
            pi.pargs.kunbus_catalog,
        )

    else:
        # No subcommand used, this will start up the server (default)
        import signal

        root = RevPiUaServerManager()

        # Set signals
        signal.signal(signal.SIGUSR1, lambda n, f: root.rotate_logfile())
        signal.signal(signal.SIGINT, lambda n, f: root.stop())
        signal.signal(signal.SIGTERM, lambda n, f: root.stop())

        return_code = root.start()

    pi.cleanup()
    return return_code
