# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2023-2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""Test MQTT system with a broker."""
from logging import getLogger, DEBUG
from time import sleep

from revpimodio2 import RevPiModIOSelected
from revpimodio2.modio import DevSelect

from mqtt_revpi_client.client_mqtt import MqttClient
from _helpers import TestCaseMqtt

log = getLogger()
log.setLevel(DEBUG)


class MqttClientBroker(TestCaseMqtt):
    """Test client_mqtt.py with a broker."""

    pictory_file = "broker_commands_tests.rsc"

    def send_commands(self, output_payload="1"):
        self.mqtt.publish("dev/all/set/O_1", output_payload)
        self.mqtt.publish("dev/all/reset/Counter_1")

        # Give broker and MqttClient some time to process the topics
        sleep(1.0)

        # Read process image to work with changed IOs
        self.revpi_simulator.readprocimg()

    def test_commands_read_only(self):

        # Prepare the counter for reset command
        self.revpi_simulator.io.Counter_1.value = 3
        self.revpi_simulator.writeprocimg()

        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
        )
        self.assertEqual(2, len(rpi.device))

        # Use "read only" device
        self.client = MqttClient(rpi.device.export_marked_event_ro)
        self.client.start()
        self.client.startup_complete.wait()

        self.send_commands()

        # On a "read only" device nothing must happen
        self.assertFalse(self.revpi_simulator.io.O_1.value)
        self.assertEqual(3, self.revpi_simulator.io.Counter_1.value)

    def test_commands_read_and_write(self):

        # Prepare the counter for reset command
        self.revpi_simulator.io.Counter_1.value = 3
        self.revpi_simulator.writeprocimg()

        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
        )
        self.assertEqual(2, len(rpi.device))

        # Use "read and write" device
        self.create_client(rpi.device.export_marked_event_rw)
        self.send_commands()

        # On a "read and write" device the values must be changed
        self.assertTrue(self.revpi_simulator.io.O_1.value)
        self.assertEqual(0, self.revpi_simulator.io.Counter_1.value)

        # Test other payload values, which should be converted as bool
        self.send_commands(output_payload="false")
        self.assertFalse(self.revpi_simulator.io.O_1.value)
        self.send_commands(output_payload="true")
        self.assertTrue(self.revpi_simulator.io.O_1.value)
