# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2023-2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""Tests for the client_mqtt.py module."""
import unittest
from logging import getLogger
from os.path import dirname, abspath, join
from tempfile import NamedTemporaryFile

from revpimodio2 import RevPiModIOSelected
from revpimodio2.modio import DevSelect

from mqtt_revpi_client import client_mqtt
from mqtt_revpi_client.client_mqtt import SharedIos

log = getLogger(__name__)
BASE_PATH = dirname(abspath(__file__))
SERVER_JOIN_TIME = 10


class MqttClientBasics(unittest.TestCase):
    """
    Test functions for the instantiation of the MqttClient class - no start.

    Various PiCtory configurations for instantiation are tested. In the case
    of valid configurations, the recognized values are checked. In the case of
    incorrect instantiations, the error handling of the class is tested.
    """

    def setUp(self) -> None:
        # Prepare empty process image
        self.procimg = NamedTemporaryFile("rb+", 0)
        self.procimg.write(b"\x00" * 4096)

    def tearDown(self) -> None:
        # Close process image file, this will delete it as well
        self.procimg.close()

    def test_valid_pictory(self):
        """
        Test a valid PiCtory configuration.

        Diese Konfiguration hat auf Position 64 einen virtuellen MQTT Client.
        Die Konfiguration besteht aus einem AIO, Core SE, DIO, MQTT Client.
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=join(BASE_PATH, "pictory_configs/aio_corese_dio_mqtt.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)
        client_mqtt.MqttClient(rpi.device.mqtt_default_settings)

    def test_get_properies(self):
        """Test a valid PiCtory configuration from test_valid_pictory."""
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=join(BASE_PATH, "pictory_configs/aio_corese_dio_mqtt.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)

        client = client_mqtt.MqttClient(rpi.device.mqtt_default_settings)
        self.assertTrue(client.broker_address, "127.0.0.1")
        self.assertFalse(client.connected)

    def test_invalid_device_id(self):
        """Configuration has an unknown device ID.

        The MQTT client device has an unknown device ID. We set it to
        '20231024_0_0' instead of '20231024_1_0'.
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=join(BASE_PATH, "pictory_configs/invalid_deviceid.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 1)

        with self.assertRaisesRegex(
            RuntimeError,
            "The device version 'device_MQTTRevPiClient_20231024_0_0_001' "
            "of mqtt client is not supported",
        ):
            client_mqtt.MqttClient(rpi.device.mqtt_device_invalid_id)

    def test_shared_ios_values(self):
        """
        Check the enum values between PiCtory and the mqtt client.

        The PiCtory configuration has some MQTT client devices. Position 64
        exports all existing IOs, position 65 exports only the exported IOs.

        Position 64, 65, 66 = MQTTRevPiClient_20231024_1_0
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=join(BASE_PATH, "pictory_configs/corese_mqtt_devices.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 3)

        client = client_mqtt.MqttClient(rpi.device.export_all_cyclic_5_rw)
        self.assertIs(client._shared_ios, SharedIos.ALL)

        client = client_mqtt.MqttClient(rpi.device.export_marked_event_ro)
        self.assertIs(client._shared_ios, SharedIos.EXPORTED)

    def test_sending_behavior(self):
        """
        Test values of sending behavior.

        Test for special values on sending behavior in the RAP file. The test
        config.rsc has some MQTT clients with various settings.
            Position 64: Sending_behavior = "Send every 5 second"
            Position 65: Sending_behavior = "Send on change"
            Position 66: Sending_behavior = "Request with base_topic/get"

        This is the enum of the RAP file:
            "Request with base_topic/get|0",
            "Send on change|255",
            "Send every 1 second|1",
            "Send every 5 second|5",
            "Send every 10 second|10",
            "Send every 15 second|15",
            "Send every 30 second|30",
            "Send every 60 second|60"
        """
        rpi = RevPiModIOSelected(
            DevSelect(search_key="productType", search_values=("24586",)),
            configrsc=join(BASE_PATH, "pictory_configs/corese_mqtt_devices.rsc"),
            procimg=self.procimg.name,
        )
        self.assertEqual(len(rpi.device), 3)

        export_all_cyclic_5_rw = client_mqtt.MqttClient(rpi.device.export_all_cyclic_5_rw)
        export_marked_event_ro = client_mqtt.MqttClient(rpi.device.export_marked_event_ro)
        export_marked_request_ro = client_mqtt.MqttClient(rpi.device.export_marked_request_ro)

        self.assertFalse(export_all_cyclic_5_rw._send_events)
        self.assertEqual(export_all_cyclic_5_rw._sendinterval, 5)
        self.assertTrue(export_marked_event_ro._send_events)
        self.assertEqual(export_marked_event_ro._sendinterval, 0)
        self.assertFalse(export_marked_request_ro._send_events)
        self.assertEqual(export_marked_request_ro._sendinterval, 0)
