# -*- coding: utf-8 -*-
# SPDX-FileCopyrightText: 2023-2024 KUNBUS GmbH
# SPDX-License-Identifier: GPL-2.0-or-later
"""Shared helpers for all tests."""

import unittest
from collections import namedtuple
from logging import getLogger
from os.path import dirname, abspath, join
from queue import Queue
from tempfile import NamedTemporaryFile
from threading import Event
from time import sleep

from paho.mqtt.client import Client, MQTTMessage
from revpimodio2 import RevPiModIO
from revpimodio2.device import Virtual

from mqtt_revpi_client.client_mqtt import MqttClient

log = getLogger(__name__)
BASE_PATH = dirname(abspath(__file__))

MqttResult = namedtuple("MqttResult", ["topic", "value"])


class TestCaseMqtt(unittest.TestCase):
    """Test client_mqtt.py with a broker."""

    pictory_file = ""

    def _on_connect(self, client: Client, userdata, flags, rc):
        """MQTT callback for established connections."""
        self.assertEqual(0, rc)
        client.subscribe("dev/+/io/#", qos=2)
        self.event_mqtt_connected.set()

    def _on_message(self, client, userdata, msg: MQTTMessage) -> None:
        """MQTT callback for a received message."""
        self.message_queue.put_nowait(
            MqttResult(
                msg.topic,
                int(msg.payload.decode()),
            )
        )

    def create_client(self, virtual_device: Virtual) -> None:
        """Create a connected client in self.client."""
        self.client = MqttClient(virtual_device, revpimodio_debugging=True)
        self.client.start()
        self.client.startup_complete.wait()

        # Wait max 5 seconds to connect client to broker, with subscriptions
        timeout = 5
        while not self.client.connected:
            sleep(1)
            timeout -= 1
            if timeout == 0:
                raise RuntimeError("MqttClient does not connect to broker")

    def setUp(self) -> None:
        self.client = None  # type: MqttClient
        self.config_rsc = join(BASE_PATH, "pictory_configs", self.pictory_file)
        self.event_mqtt_connected = Event()
        self.message_queue = Queue()

        # Prepare empty process image
        self.procimg = NamedTemporaryFile("rb+", 0)
        self.procimg.write(b"\x00" * 4096)

        # Set up the ModIO Simulator
        self.revpi_simulator = RevPiModIO(
            simulator=True,
            configrsc=self.config_rsc,
            procimg=self.procimg.name,
            shared_procimg=True,
        )
        self.revpi_simulator.setdefaultvalues()
        self.revpi_simulator.writeprocimg()

        self.mqtt = Client()
        self.mqtt.on_connect = self._on_connect
        self.mqtt.on_message = self._on_message
        self.mqtt.connect_async(host="127.0.0.1", port=51883)
        self.mqtt.loop_start()
        self.event_mqtt_connected.wait(15.0)

    def tearDown(self) -> None:
        # Tear down created client to prevent deadlocks
        if self.client:
            self.client.stop()

        self.mqtt.disconnect()
        self.revpi_simulator.cleanup()
        self.procimg.close()
